<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['REQUESTER']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}

$hasCategories = table_exists($pdo, 'categories');

if ($hasCategories) {
  $items = $pdo->query("
    SELECT i.id, i.name, i.unit, i.stock_qty, c.name AS category_name
    FROM items i
    LEFT JOIN categories c ON c.id = i.category_id
    WHERE i.is_active=1
    ORDER BY c.name, i.name
  ")->fetchAll();
} else {
  $items = $pdo->query("
    SELECT id, name, unit, stock_qty
    FROM items
    WHERE is_active=1
    ORDER BY name
  ")->fetchAll();
}

include __DIR__ . '/../partials/header.php';
?>

<h3 class="mb-3">New Request</h3>

<form method="post" action="<?=h(base_url('requester/actions/submit_request.php'))?>" class="card shadow-sm">
  <div class="card-body">

    <div class="row g-2 align-items-end">
      <div class="col-md-7">
        <label class="form-label fw-semibold">Search item</label>

        <!-- Searchable input with browser auto-suggest -->
        <input class="form-control" id="item_search" list="items_list" placeholder="Type to search (category, name...)">
        <datalist id="items_list">
          <?php foreach ($items as $it): ?>
            <?php
              $cat = $it['category_name'] ?? '';
              $label = ($cat ? ($cat . ' — ') : '') . $it['name'] . " (Stock: " . $it['stock_qty'] . " " . $it['unit'] . ")";
            ?>
            <option value="<?=h($label)?>"></option>
          <?php endforeach; ?>
        </datalist>

        <input type="hidden" id="item_id" value="">
        <div class="form-text">Start typing and pick from the dropdown suggestions.</div>
      </div>

      <div class="col-md-3">
        <label class="form-label fw-semibold">Qty</label>
        <input class="form-control" id="qty" type="number" min="1" value="1">
      </div>

      <div class="col-md-2">
        <button type="button" class="btn btn-outline-primary w-100" onclick="addLine()">Add</button>
      </div>
    </div>

    <hr class="my-4">

    <div class="table-responsive">
      <table class="table table-hover align-middle" id="linesTable">
        <thead class="table-light">
          <tr>
            <th>Item</th>
            <th style="width:140px;">Qty</th>
            <th style="width:110px;"></th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    </div>

    <div class="mb-3">
      <label class="form-label fw-semibold">Note</label>
      <textarea class="form-control" name="note" rows="2" placeholder="Optional note..."></textarea>
    </div>

    <div class="d-flex gap-2">
      <button class="btn btn-success" type="submit">Submit Request</button>
      <a class="btn btn-outline-secondary" href="<?=h(base_url('requester/my_requests.php'))?>">Cancel</a>
    </div>

    <div id="linesInputs"></div>
  </div>
</form>

<script>
// Map: label -> id   and  id -> display name
const labelToId = new Map();
const idToDisplay = new Map();

<?php foreach ($items as $it): ?>
<?php
  $cat = $it['category_name'] ?? '';
  $label = ($cat ? ($cat . ' — ') : '') . $it['name'] . " (Stock: " . $it['stock_qty'] . " " . $it['unit'] . ")";
  $display = ($cat ? ($cat . ' — ') : '') . $it['name'] . " (" . $it['unit'] . ")";
?>
labelToId.set("<?=h(addslashes($label))?>", "<?=h($it['id'])?>");
idToDisplay.set("<?=h($it['id'])?>", "<?=h(addslashes($display))?>");
<?php endforeach; ?>

// Keep hidden item_id synced when user chooses a suggestion
const searchInput = document.getElementById('item_search');
const hiddenItemId = document.getElementById('item_id');

function syncSelectedId() {
  const v = searchInput.value || '';
  const id = labelToId.get(v);
  hiddenItemId.value = id ? id : '';
}

searchInput.addEventListener('input', syncSelectedId);
searchInput.addEventListener('change', syncSelectedId);

function addLine() {
  syncSelectedId();

  const itemId = hiddenItemId.value;
  const qty = parseInt(document.getElementById('qty').value || '0', 10);

  if (!itemId) {
    alert('Please choose an item from the dropdown suggestions.');
    return;
  }
  if (qty < 1) return;

  const existing = document.querySelector(`#linesTable tbody tr[data-item-id="${itemId}"]`);
  if (existing) {
    const qInput = existing.querySelector('input.line-qty');
    qInput.value = parseInt(qInput.value, 10) + qty;
    syncHiddenInputs();
    return;
  }

  const tr = document.createElement('tr');
  tr.dataset.itemId = itemId;
  tr.innerHTML = `
    <td class="fw-semibold">${idToDisplay.get(itemId) || itemId}</td>
    <td><input type="number" min="1" class="form-control line-qty" value="${qty}" onchange="syncHiddenInputs()"></td>
    <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="removeLine('${itemId}')">Remove</button></td>
  `;

  document.querySelector('#linesTable tbody').appendChild(tr);

  // UX: reset search input for next item
  searchInput.value = '';
  hiddenItemId.value = '';
  document.getElementById('qty').value = 1;

  syncHiddenInputs();
}

function removeLine(itemId) {
  const tr = document.querySelector(`#linesTable tbody tr[data-item-id="${itemId}"]`);
  if (tr) tr.remove();
  syncHiddenInputs();
}

function syncHiddenInputs() {
  const container = document.getElementById('linesInputs');
  container.innerHTML = '';
  const rows = document.querySelectorAll('#linesTable tbody tr');
  let i = 0;
  rows.forEach(r => {
    const itemId = r.dataset.itemId;
    const qty = r.querySelector('input.line-qty').value;
    container.insertAdjacentHTML('beforeend', `
      <input type="hidden" name="lines[${i}][item_id]" value="${itemId}">
      <input type="hidden" name="lines[${i}][qty]" value="${qty}">
    `);
    i++;
  });
}
</script>

<?php include __DIR__ . '/../partials/footer.php'; ?>
